const API_BASE = '../api';
let currentConfig = {};

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadSettings();
    loadMenu();
    
    document.getElementById('shopSettingsForm').addEventListener('submit', saveSettings);
    document.getElementById('itemForm').addEventListener('submit', saveMenuItem);
});

// Load Settings
async function loadSettings() {
    try {
        const response = await fetch(`${API_BASE}/load_settings.php`);
        const data = await response.json();
        
        if (data.success) {
            currentConfig = data.config;
            
            // Populate form
            document.getElementById('shopName').value = data.config.shop_name;
            document.getElementById('taxPercent').value = data.config.tax_percent;
            
            // Update package info
            document.getElementById('currentPlan').textContent = data.config.plan.toUpperCase();
            document.getElementById('ordersRemaining').textContent = data.config.package_orders_remaining;
            
            // Render packages
            renderPackages(data.packages);
        }
    } catch (error) {
        console.error('Load settings error:', error);
    }
}

// Save Settings
async function saveSettings(e) {
    e.preventDefault();
    
    const shopName = document.getElementById('shopName').value;
    const taxPercent = document.getElementById('taxPercent').value;
    
    try {
        const response = await fetch(`${API_BASE}/update_settings.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                shop_name: shopName,
                tax_percent: parseFloat(taxPercent)
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            alert('Settings saved successfully!');
            loadSettings();
        } else {
            alert('Failed to save settings');
        }
    } catch (error) {
        console.error('Save settings error:', error);
        alert('Failed to save settings');
    }
}

// Render Packages
function renderPackages(packages) {
    const packagesList = document.getElementById('packagesList');
    packagesList.innerHTML = packages.map(pkg => `
        <div class="package-card">
            <h3>${pkg.name}</h3>
            <div class="package-orders">${pkg.orders_count} Orders</div>
            <div class="package-price">$${parseFloat(pkg.price).toFixed(2)}</div>
            <p class="package-description">${pkg.description}</p>
            <button class="btn btn-primary btn-block" onclick="purchasePackage(${pkg.id})">
                Purchase
            </button>
        </div>
    `).join('');
}

// Purchase Package
async function purchasePackage(packageId) {
    if (!confirm('Are you sure you want to purchase this package?')) {
        return;
    }
    
    try {
        const response = await fetch(`${API_BASE}/purchase_package.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ package_id: packageId })
        });
        
        const data = await response.json();
        
        if (data.success) {
            alert(data.message);
            loadSettings();
        } else {
            alert('Failed to purchase package: ' + data.error);
        }
    } catch (error) {
        console.error('Purchase error:', error);
        alert('Failed to purchase package');
    }
}

// Load Menu
async function loadMenu() {
    try {
        const response = await fetch(`${API_BASE}/manage_menu.php`);
        const data = await response.json();
        
        if (data.success) {
            renderMenu(data.menu);
        }
    } catch (error) {
        console.error('Load menu error:', error);
    }
}

// Render Menu
function renderMenu(menu) {
    const menuList = document.getElementById('menuList');
    menuList.innerHTML = menu.map(item => `
        <div class="menu-manage-item ${!item.active ? 'inactive' : ''}">
            <img src="${item.image}" alt="${item.name}" class="menu-manage-image">
            <div class="menu-manage-info">
                <div class="menu-manage-name">${item.name}</div>
                <div class="menu-manage-category">${item.category}</div>
                <div class="menu-manage-price">$${parseFloat(item.price).toFixed(2)}</div>
                <div class="menu-manage-status">${item.active ? 'Active' : 'Inactive'}</div>
            </div>
            <div class="menu-manage-actions">
                <button class="btn btn-small btn-primary" onclick='editMenuItem(${JSON.stringify(item)})'>Edit</button>
                <button class="btn btn-small btn-remove" onclick="deleteMenuItem(${item.id})">Delete</button>
            </div>
        </div>
    `).join('');
}

// Show Add Modal
function showAddItemModal() {
    document.getElementById('modalTitle').textContent = 'Add Menu Item';
    document.getElementById('itemForm').reset();
    document.getElementById('itemId').value = '';
    document.getElementById('itemModal').style.display = 'block';
}

// Edit Menu Item
function editMenuItem(item) {
    document.getElementById('modalTitle').textContent = 'Edit Menu Item';
    document.getElementById('itemId').value = item.id;
    document.getElementById('itemName').value = item.name;
    document.getElementById('itemCategory').value = item.category;
    document.getElementById('itemPrice').value = item.price;
    document.getElementById('itemImage').value = item.image;
    document.getElementById('itemActive').checked = item.active == 1;
    document.getElementById('itemModal').style.display = 'block';
}

// Save Menu Item
async function saveMenuItem(e) {
    e.preventDefault();
    
    const itemId = document.getElementById('itemId').value;
    const itemData = {
        name: document.getElementById('itemName').value,
        category: document.getElementById('itemCategory').value,
        price: parseFloat(document.getElementById('itemPrice').value),
        image: document.getElementById('itemImage').value,
        active: document.getElementById('itemActive').checked
    };
    
    try {
        let response;
        if (itemId) {
            // Update
            itemData.id = parseInt(itemId);
            response = await fetch(`${API_BASE}/manage_menu.php`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(itemData)
            });
        } else {
            // Create
            response = await fetch(`${API_BASE}/manage_menu.php`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(itemData)
            });
        }
        
        const data = await response.json();
        
        if (data.success) {
            alert('Menu item saved successfully!');
            closeModal();
            loadMenu();
        } else {
            alert('Failed to save menu item');
        }
    } catch (error) {
        console.error('Save menu item error:', error);
        alert('Failed to save menu item');
    }
}

// Delete Menu Item
async function deleteMenuItem(itemId) {
    if (!confirm('Are you sure you want to delete this item?')) {
        return;
    }
    
    try {
        const response = await fetch(`${API_BASE}/manage_menu.php?id=${itemId}`, {
            method: 'DELETE'
        });
        
        const data = await response.json();
        
        if (data.success) {
            alert('Menu item deleted successfully!');
            loadMenu();
        } else {
            alert('Failed to delete menu item');
        }
    } catch (error) {
        console.error('Delete error:', error);
        alert('Failed to delete menu item');
    }
}

// Close Modal
function closeModal() {
    document.getElementById('itemModal').style.display = 'none';
}

// Close modal on outside click
window.onclick = function(event) {
    const modal = document.getElementById('itemModal');
    if (event.target == modal) {
        closeModal();
    }
}