<?php
require_once 'config.php';

$conn = getDB();

$data = json_decode(file_get_contents('php://input'), true);

$table = $conn->real_escape_string($data['table']);
$items = $data['items'];
$orderGroupId = uniqid('order_');
$orderDate = date('Y-m-d');

if (empty($table) || empty($items)) {
    echo json_encode(['success' => false, 'error' => 'Invalid data']);
    exit;
}

// Get tax percent
$configResult = $conn->query("SELECT tax_percent, package_orders_remaining FROM config LIMIT 1");
$config = $configResult->fetch_assoc();
$taxPercent = floatval($config['tax_percent']);
$ordersRemaining = intval($config['package_orders_remaining']);

// Check if orders remaining (if using package plan)
if ($ordersRemaining <= 0 && $config['plan'] === 'package') {
    echo json_encode(['success' => false, 'error' => 'No orders remaining. Please purchase a package.']);
    exit;
}

$conn->begin_transaction();

try {
    $stmt = $conn->prepare("
        INSERT INTO orders (order_date, table_name, item_id, item_name, qty, price, subtotal, tax_amount, total, status, order_group_id) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'ongoing', ?)
    ");

    foreach ($items as $item) {
        $itemId = $item['item_id'];
        $itemName = $item['item_name'];
        $qty = $item['qty'];
        $price = $item['price'];
        $subtotal = $item['subtotal'];
        $taxAmount = $subtotal * ($taxPercent / 100);
        $total = $subtotal + $taxAmount;
        
        $stmt->bind_param('ssissdddds', $orderDate, $table, $itemId, $itemName, $qty, $price, $subtotal, $taxAmount, $total, $orderGroupId);
        $stmt->execute();
    }

    // Update table status to in-service
    $conn->query("
        INSERT INTO tables (name, status) VALUES ('$table', 'in-service')
        ON DUPLICATE KEY UPDATE status = 'in-service'
    ");
    
    // Decrement package orders if using package plan
    if ($ordersRemaining > 0) {
        $conn->query("UPDATE config SET package_orders_remaining = package_orders_remaining - 1");
    }

    $conn->commit();
    
    echo json_encode(['success' => true, 'order_group_id' => $orderGroupId]);
    
} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

$conn->close();
?>